// متغيرات عامة
let extractedEntries = [];
let isExtractingInProgress = false;

// تهيئة الإضافة عند التحميل
document.addEventListener('DOMContentLoaded', async () => {
    showNotification('مرحباً بك في مستخرج Entry IDs - حميز نت', 'success');
    await checkCurrentPage();
    
    // ربط أحداث الأزرار
    setupEventListeners();
});

// إعداد مستمعي الأحداث
function setupEventListeners() {
    // زر الاستخراج
    document.getElementById('extractBtn').addEventListener('click', extractEntries);
    
    // زر التحديث
    document.getElementById('refreshBtn').addEventListener('click', refreshData);
    
    // زر نسخ الكل
    document.getElementById('copyAllBtn').addEventListener('click', copyAllEntries);
    
    // أزرار التصدير
    document.getElementById('exportTextBtn').addEventListener('click', exportAsText);
    document.getElementById('exportJSONBtn').addEventListener('click', exportAsJSON);
    document.getElementById('exportCSVBtn').addEventListener('click', exportAsCSV);
    
    // أزرار محول الروابط
    document.getElementById('detectUrlBtn').addEventListener('click', detectCurrentUrl);
    document.getElementById('copyUrlBtn').addEventListener('click', copyConvertedUrl);
    
    // مراقبة تغيير الرابط المدخل
    document.getElementById('formUrl').addEventListener('input', handleUrlInput);
}

// فحص الصفحة الحالية
async function checkCurrentPage() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        
        if (!tab.url.includes('docs.google.com/forms')) {
            updateStatus('error', 'يرجى فتح صفحة Google Forms أولاً');
            document.getElementById('extractBtn').disabled = true;
            return;
        }

        updateStatus('success', 'تم اكتشاف صفحة Google Forms - جاهز للاستخراج');
        document.getElementById('extractBtn').disabled = false;
        
        // كشف الرابط تلقائياً
        detectCurrentUrl();
        
        // استخراج تلقائي عند فتح الإضافة
        setTimeout(() => {
            extractEntries();
        }, 500);
        
    } catch (error) {
        updateStatus('error', 'خطأ في فحص الصفحة');
        console.error('Error checking page:', error);
    }
}

// كشف رابط الصفحة الحالية
async function detectCurrentUrl() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        const urlInput = document.getElementById('formUrl');
        
        if (tab.url.includes('docs.google.com/forms')) {
            urlInput.value = tab.url;
            handleUrlInput();
            showNotification('تم كشف رابط النموذج تلقائياً', 'success');
        } else {
            showNotification('لم يتم العثور على رابط Google Forms', 'warning');
        }
    } catch (error) {
        console.error('Error detecting URL:', error);
        showNotification('خطأ في كشف الرابط', 'error');
    }
}

// معالجة إدخال الرابط
function handleUrlInput() {
    const urlInput = document.getElementById('formUrl');
    const conversionResult = document.getElementById('conversionResult');
    const convertedUrlSpan = document.getElementById('convertedUrl');
    
    const inputUrl = urlInput.value.trim();
    
    if (!inputUrl) {
        conversionResult.style.display = 'none';
        return;
    }
    
    // التحقق من صحة الرابط
    if (!inputUrl.includes('docs.google.com/forms')) {
        conversionResult.style.display = 'none';
        showNotification('الرابط المدخل ليس رابط Google Forms صحيح', 'warning');
        return;
    }
    
    // استخراج معرف النموذج
    const formIdMatch = inputUrl.match(/\/forms\/d\/e\/([a-zA-Z0-9-_]+)/);
    if (!formIdMatch) {
        conversionResult.style.display = 'none';
        showNotification('لم يتم العثور على معرف النموذج في الرابط', 'error');
        return;
    }
    
    const formId = formIdMatch[1];
    const responseUrl = `https://docs.google.com/forms/d/e/${formId}/formResponse`;
    
    convertedUrlSpan.textContent = responseUrl;
    conversionResult.style.display = 'block';
    
    showNotification('تم تحويل الرابط بنجاح!', 'success');
}

// نسخ الرابط المحول
async function copyConvertedUrl() {
    const convertedUrl = document.getElementById('convertedUrl').textContent;
    
    if (!convertedUrl) {
        showNotification('لا يوجد رابط للنسخ', 'warning');
        return;
    }
    
    try {
        if (navigator.clipboard && window.isSecureContext) {
            await navigator.clipboard.writeText(convertedUrl);
        } else {
            await fallbackCopyTextToClipboard(convertedUrl);
        }
        
        // تحديث زر النسخ
        const copyUrlBtn = document.getElementById('copyUrlBtn');
        const originalHTML = copyUrlBtn.innerHTML;
        const originalBgColor = copyUrlBtn.style.backgroundColor;
        
        copyUrlBtn.innerHTML = '<i class="fas fa-check"></i> تم النسخ';
        copyUrlBtn.style.backgroundColor = '#22c55e';
        copyUrlBtn.disabled = true;
        
        setTimeout(() => {
            copyUrlBtn.innerHTML = originalHTML;
            copyUrlBtn.style.backgroundColor = originalBgColor;
            copyUrlBtn.disabled = false;
        }, 2000);
        
        showNotification('تم نسخ رابط الاستجابة بنجاح!', 'success');
        
    } catch (error) {
        console.error('Copy URL error:', error);
        showNotification('فشل في نسخ الرابط', 'error');
    }
}

// استخراج Entry IDs
async function extractEntries() {
    if (isExtractingInProgress) return;
    
    isExtractingInProgress = true;
    showLoading(true);
    updateStatus('loading', 'جاري استخراج Entry IDs...');
    
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        
        // حقن سكريبت الاستخراج في الصفحة
        const results = await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            function: extractGoogleFormEntries
        });
        
        if (results && results[0] && results[0].result) {
            extractedEntries = results[0].result;
            
            if (extractedEntries.length > 0) {
                displayResults();
                updateStatus('success', `تم استخراج ${extractedEntries.length} حقل بنجاح`);
                showNotification(`تم استخراج ${extractedEntries.length} حقل بنجاح!`, 'success');
            } else {
                showError();
                updateStatus('error', 'لم يتم العثور على أي حقول Entry');
            }
        } else {
            showError();
            updateStatus('error', 'فشل في استخراج البيانات');
        }
        
    } catch (error) {
        console.error('Extraction error:', error);
        showError();
        updateStatus('error', 'حدث خطأ أثناء الاستخراج');
        showNotification('حدث خطأ أثناء الاستخراج', 'error');
    } finally {
        showLoading(false);
        isExtractingInProgress = false;
    }
}

// دالة الاستخراج الرئيسية (يتم حقنها في الصفحة)
function extractGoogleFormEntries() {
    let entryData = [];
    let foundEntries = false;
    
    console.log("🔍 بدء عملية البحث عن حقول النموذج...");
    
    // الطريقة الأولى: البحث في data-params
    const elementsWithDataParams = document.querySelectorAll('[data-params*="entry."]');
    
    if (elementsWithDataParams.length > 0) {
        console.log(`✅ تم العثور على ${elementsWithDataParams.length} حقل باستخدام data-params`);
        
        elementsWithDataParams.forEach((element, index) => {
            try {
                // استخراج اسم الحقل
                const fieldNameElement = element.querySelector('.M7eMe, [role="heading"]');
                let fieldName = fieldNameElement ? fieldNameElement.textContent.trim() : null;
                
                // إذا لم نجد الاسم، ابحث في العنصر نفسه
                if (!fieldName) {
                    const spans = element.querySelectorAll('span');
                    for (let span of spans) {
                        if (span.textContent && span.textContent.trim() && !span.textContent.includes('entry.')) {
                            fieldName = span.textContent.trim();
                            break;
                        }
                    }
                }
                
                // استخراج Entry ID من data-params
                const dataParams = element.getAttribute('data-params');
                const entryMatches = dataParams.match(/\[(\d{9,}),/g);
                
                if (entryMatches) {
                    entryMatches.forEach((match, matchIndex) => {
                        const entryId = match.replace(/[\[\,]/g, '');
                        if (entryId && entryId.length >= 9) {
                            const finalFieldName = fieldName || `حقل ${index + 1}-${matchIndex + 1}`;
                            entryData.push({
                                fieldName: finalFieldName,
                                entryId: entryId,
                                fullEntry: `entry.${entryId}`,
                                method: 'data-params'
                            });
                            foundEntries = true;
                        }
                    });
                }
            } catch (error) {
                console.warn(`⚠️ خطأ في معالجة العنصر ${index + 1}:`, error);
            }
        });
    }
    
    // الطريقة الثانية: البحث في name attributes
    const formFields = document.querySelectorAll('[name^="entry."]');
    
    if (formFields.length > 0) {
        console.log(`✅ تم العثور على ${formFields.length} حقل باستخدام name attributes`);
        
        formFields.forEach((field, index) => {
            try {
                const entryId = field.name.replace('entry.', '');
                
                // البحث عن اسم الحقل في العناصر المجاورة
                let fieldName = null;
                
                // البحث في العنصر الأب
                let parent = field.closest('[role="listitem"], .freebirdFormviewerViewItemsItemItem');
                if (parent) {
                    const nameElement = parent.querySelector('.M7eMe, [role="heading"], .freebirdFormviewerViewItemsItemItemTitle');
                    if (nameElement) {
                        fieldName = nameElement.textContent.trim();
                    }
                }
                
                // إذا لم نجد الاسم، استخدم placeholder أو label
                if (!fieldName) {
                    fieldName = field.placeholder || field.getAttribute('aria-label') || `حقل ${index + 1}`;
                }
                
                // تحقق من عدم وجود هذا Entry ID مسبقاً
                const exists = entryData.some(item => item.entryId === entryId);
                if (!exists) {
                    entryData.push({
                        fieldName: fieldName,
                        entryId: entryId,
                        fullEntry: `entry.${entryId}`,
                        method: 'name-attribute'
                    });
                    foundEntries = true;
                }
            } catch (error) {
                console.warn(`⚠️ خطأ في معالجة الحقل ${index + 1}:`, error);
            }
        });
    }
    
    // الطريقة الثالثة: البحث في النص المصدري للصفحة
    const pageHTML = document.documentElement.outerHTML;
    const entryMatches = pageHTML.match(/entry\.(\d{9,})/g);
    
    if (entryMatches) {
        const uniqueEntries = [...new Set(entryMatches)];
        console.log(`✅ تم العثور على ${uniqueEntries.length} entry في النص المصدري`);
        
        uniqueEntries.forEach((entry, index) => {
            const entryId = entry.replace('entry.', '');
            
            // تحقق من عدم وجود هذا Entry ID مسبقاً
            const exists = entryData.some(item => item.entryId === entryId);
            if (!exists) {
                entryData.push({
                    fieldName: `حقل مكتشف ${index + 1}`,
                    entryId: entryId,
                    fullEntry: entry,
                    method: 'source-code'
                });
                foundEntries = true;
            }
        });
    }
    
    // البحث الشامل كمحاولة أخيرة
    if (!foundEntries || entryData.length === 0) {
        console.log("🔄 محاولة البحث الشامل...");
        const allElements = document.querySelectorAll('*');
        
        allElements.forEach(element => {
            // البحث في جميع الخصائص
            for (let attr of element.attributes || []) {
                if (attr.value && attr.value.includes('entry.')) {
                    const matches = attr.value.match(/entry\.(\d{9,})/g);
                    if (matches) {
                        matches.forEach(match => {
                            const entryId = match.replace('entry.', '');
                            const exists = entryData.some(item => item.entryId === entryId);
                            if (!exists) {
                                entryData.push({
                                    fieldName: `حقل من ${attr.name}`,
                                    entryId: entryId,
                                    fullEntry: match,
                                    method: 'attribute-search'
                                });
                                foundEntries = true;
                            }
                        });
                    }
                }
            }
        });
    }
    
    // إزالة المكررات وترتيب البيانات
    entryData = entryData.filter((item, index, self) => 
        index === self.findIndex(t => t.entryId === item.entryId)
    );
    
    console.log(`📊 تم استخراج ${entryData.length} حقل في المجموع`);
    
    return entryData;
}

// عرض النتائج
function displayResults() {
    const resultsSection = document.getElementById('resultsSection');
    const entriesList = document.getElementById('entriesList');
    const totalCount = document.getElementById('totalCount');
    const errorSection = document.getElementById('errorSection');
    
    // إخفاء قسم الخطأ وإظهار النتائج
    errorSection.style.display = 'none';
    resultsSection.style.display = 'block';
    
    // تحديث العدد الكلي
    totalCount.textContent = extractedEntries.length;
    
    // مسح القائمة السابقة
    entriesList.innerHTML = '';
    
    // إضافة كل entry إلى القائمة
    extractedEntries.forEach((entry, index) => {
        const entryElement = createEntryElement(entry, index + 1);
        entriesList.appendChild(entryElement);
    });
}

// إنشاء عنصر Entry
function createEntryElement(entry, number) {
    const entryDiv = document.createElement('div');
    entryDiv.className = 'entry-item';
    
    const methodIcon = getMethodIcon(entry.method);
    
    entryDiv.innerHTML = `
        <div class="entry-header">
            <div class="entry-number">${number}</div>
            <div class="entry-name">${entry.fieldName}</div>
            <button class="entry-copy-btn" data-entry="${entry.fullEntry}">
                <i class="fas fa-copy"></i>
            </button>
        </div>
        <div class="entry-id">${entry.fullEntry}</div>
        <div class="entry-method">
            <i class="${methodIcon}"></i>
            ${getMethodName(entry.method)}
        </div>
    `;
    
    // إضافة مستمع الحدث لزر النسخ
    const copyBtn = entryDiv.querySelector('.entry-copy-btn');
    copyBtn.addEventListener('click', function() {
        copyEntry(this.dataset.entry, this);
    });
    
    return entryDiv;
}

// الحصول على أيقونة الطريقة
function getMethodIcon(method) {
    const icons = {
        'data-params': 'fas fa-database',
        'name-attribute': 'fas fa-tag',
        'source-code': 'fas fa-code',
        'attribute-search': 'fas fa-search'
    };
    return icons[method] || 'fas fa-question';
}

// الحصول على اسم الطريقة
function getMethodName(method) {
    const names = {
        'data-params': 'مُعاملات البيانات',
        'name-attribute': 'خاصية الاسم',
        'source-code': 'الكود المصدري',
        'attribute-search': 'البحث الشامل'
    };
    return names[method] || 'غير معروف';
}

// نسخ entry واحد مع دعم طرق النسخ المختلفة
async function copyEntry(entryText, button) {
    try {
        // محاولة استخدام Clipboard API الحديث
        if (navigator.clipboard && window.isSecureContext) {
            await navigator.clipboard.writeText(entryText);
        } else {
            // استخدام الطريقة التقليدية كبديل
            await fallbackCopyTextToClipboard(entryText);
        }
        
        // تغيير النص مؤقتاً للتأكيد
        const originalHTML = button.innerHTML;
        const originalBgColor = button.style.backgroundColor;
        
        button.innerHTML = '<i class="fas fa-check"></i>';
        button.style.backgroundColor = '#22c55e';
        button.disabled = true;
        
        setTimeout(() => {
            button.innerHTML = originalHTML;
            button.style.backgroundColor = originalBgColor;
            button.disabled = false;
        }, 1500);
        
        showNotification(`تم نسخ: ${entryText}`, 'success');
        
    } catch (error) {
        console.error('Copy error:', error);
        showNotification('فشل في النسخ. يرجى المحاولة مرة أخرى', 'error');
    }
}

// طريقة النسخ البديلة للمتصفحات القديمة
function fallbackCopyTextToClipboard(text) {
    return new Promise((resolve, reject) => {
        try {
            // إنشاء عنصر input مؤقت
            const textArea = document.createElement('textarea');
            textArea.value = text;
            textArea.style.position = 'fixed';
            textArea.style.left = '-9999px';
            textArea.style.top = '-9999px';
            
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            
            // محاولة النسخ
            const successful = document.execCommand('copy');
            document.body.removeChild(textArea);
            
            if (successful) {
                resolve();
            } else {
                reject(new Error('فشل في تنفيذ أمر النسخ'));
            }
        } catch (error) {
            reject(error);
        }
    });
}

// نسخ جميع Entries
async function copyAllEntries() {
    if (extractedEntries.length === 0) {
        showNotification('لا توجد بيانات للنسخ', 'warning');
        return;
    }
    
    try {
        const allEntries = extractedEntries.map(entry => entry.fullEntry).join(', ');
        
        // محاولة استخدام Clipboard API
        if (navigator.clipboard && window.isSecureContext) {
            await navigator.clipboard.writeText(allEntries);
        } else {
            await fallbackCopyTextToClipboard(allEntries);
        }
        
        // تحديث زر النسخ الكامل
        const copyAllBtn = document.getElementById('copyAllBtn');
        const originalText = copyAllBtn.innerHTML;
        const originalBgColor = copyAllBtn.style.backgroundColor;
        
        copyAllBtn.innerHTML = '<i class="fas fa-check"></i> تم النسخ';
        copyAllBtn.style.backgroundColor = '#22c55e';
        copyAllBtn.disabled = true;
        
        setTimeout(() => {
            copyAllBtn.innerHTML = originalText;
            copyAllBtn.style.backgroundColor = originalBgColor;
            copyAllBtn.disabled = false;
        }, 2000);
        
        showNotification(`تم نسخ ${extractedEntries.length} entry بنجاح`, 'success');
        
    } catch (error) {
        console.error('Copy all error:', error);
        showNotification('فشل في النسخ. يرجى المحاولة مرة أخرى', 'error');
    }
}

// تحديث البيانات
async function refreshData() {
    await extractEntries();
}

// عرض قسم الخطأ
function showError() {
    const resultsSection = document.getElementById('resultsSection');
    const errorSection = document.getElementById('errorSection');
    
    resultsSection.style.display = 'none';
    errorSection.style.display = 'block';
}

// إظهار/إخفاء التحميل
function showLoading(show) {
    const loading = document.getElementById('loading');
    loading.style.display = show ? 'block' : 'none';
}

// تحديث حالة الأداة
function updateStatus(type, message) {
    const statusIndicator = document.getElementById('statusIndicator');
    const statusText = document.getElementById('statusText');
    
    // إزالة الفئات السابقة
    statusIndicator.classList.remove('success', 'error', 'loading');
    
    // إضافة الفئة الجديدة
    if (type !== 'default') {
        statusIndicator.classList.add(type);
    }
    
    // تحديث النص
    statusText.textContent = message;
    
    // تحديث الأيقونة
    const icon = statusIndicator.querySelector('i');
    const icons = {
        success: 'fas fa-check-circle',
        error: 'fas fa-exclamation-circle',
        loading: 'fas fa-spinner fa-spin',
        default: 'fas fa-search'
    };
    
    icon.className = icons[type] || icons.default;
}

// إظهار إشعار
function showNotification(message, type = 'success') {
    const notification = document.getElementById('notification');
    
    notification.textContent = message;
    notification.className = `notification ${type}`;
    notification.classList.add('show');
    
    setTimeout(() => {
        notification.classList.remove('show');
    }, 3000);
}

// تصدير كنص
function exportAsText() {
    if (extractedEntries.length === 0) {
        showNotification('لا توجد بيانات للتصدير', 'warning');
        return;
    }
    
    try {
    let content = `معرفات حقول نموذج Google Forms (Entry IDs)\n`;
    content += `=${'='.repeat(50)}\n`;
    content += `📅 تاريخ التصدير: ${new Date().toLocaleString('ar-SA')}\n`;
    content += `📊 عدد الحقول: ${extractedEntries.length}\n`;
    content += `=${'='.repeat(50)}\n\n`;
    
    extractedEntries.forEach((entry, index) => {
        content += `📋 الحقل ${index + 1}: ${entry.fieldName}\n`;
        content += `🔢 Entry ID: ${entry.fullEntry}\n`;
        content += `🔍 طريقة الاكتشاف: ${getMethodName(entry.method)}\n`;
        content += `${'-'.repeat(30)}\n\n`;
    });
    
    content += `\n${'='.repeat(50)}\n`;
    content += `📋 قائمة Entry IDs للنسخ السريع:\n\n`;
    content += extractedEntries.map(entry => entry.fullEntry).join(', ');
    
    downloadFile(content, 'google_forms_entries.txt', 'text/plain');
    } catch (error) {
        console.error('Export text error:', error);
        showNotification('فشل في تصدير الملف النصي', 'error');
    }
}

// تصدير كـ JSON
function exportAsJSON() {
    if (extractedEntries.length === 0) {
        showNotification('لا توجد بيانات للتصدير', 'warning');
        return;
    }
    
    try {
    const jsonData = {
        metadata: {
            extractedAt: new Date().toISOString(),
            totalEntries: extractedEntries.length,
            extractorVersion: '1.0.0'
        },
        entries: extractedEntries
    };
    
    const content = JSON.stringify(jsonData, null, 2);
    downloadFile(content, 'google_forms_entries.json', 'application/json');
    } catch (error) {
        console.error('Export JSON error:', error);
        showNotification('فشل في تصدير ملف JSON', 'error');
    }
}

// تصدير كـ CSV
function exportAsCSV() {
    if (extractedEntries.length === 0) {
        showNotification('لا توجد بيانات للتصدير', 'warning');
        return;
    }
    
    try {
    const headers = ['الرقم', 'اسم الحقل', 'Entry ID', 'Entry كامل', 'طريقة الاكتشاف'];
    const csvRows = [headers.join(',')];
    
    extractedEntries.forEach((entry, index) => {
        const row = [
            index + 1,
            `"${entry.fieldName}"`,
            entry.entryId,
            entry.fullEntry,
            `"${getMethodName(entry.method)}"`
        ];
        csvRows.push(row.join(','));
    });
    
    const content = '\ufeff' + csvRows.join('\n'); // BOM for UTF-8
    downloadFile(content, 'google_forms_entries.csv', 'text/csv');
    } catch (error) {
        console.error('Export CSV error:', error);
        showNotification('فشل في تصدير ملف CSV', 'error');
    }
}

// تحميل ملف
function downloadFile(content, filename, mimeType) {
    try {
        const blob = new Blob([content], { type: mimeType });
        const url = URL.createObjectURL(blob);
        
        // التحقق من وجود chrome.downloads API
        if (chrome.downloads) {
            chrome.downloads.download({
                url: url,
                filename: filename,
                saveAs: true
            }, (downloadId) => {
                if (chrome.runtime.lastError) {
                    console.error('Download error:', chrome.runtime.lastError);
                    // استخدام الطريقة البديلة
                    fallbackDownload(url, filename);
                } else if (downloadId) {
                    showNotification(`تم تحميل الملف: ${filename}`, 'success');
                } else {
                    showNotification('فشل في تحميل الملف', 'error');
                }
                URL.revokeObjectURL(url);
            });
        } else {
            // استخدام الطريقة البديلة إذا لم تكن API متاحة
            fallbackDownload(url, filename);
        }
    } catch (error) {
        console.error('Download file error:', error);
        showNotification('فشل في إنشاء الملف للتحميل', 'error');
    }
}

// طريقة بديلة للتحميل
function fallbackDownload(url, filename) {
    try {
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        a.style.display = 'none';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        
        setTimeout(() => {
            URL.revokeObjectURL(url);
        }, 100);
        
        showNotification(`تم تحميل الملف: ${filename}`, 'success');
    } catch (error) {
        console.error('Fallback download error:', error);
        showNotification('فشل في تحميل الملف', 'error');
    }
}

// إضافة وظائف التصدير إلى النافذة العامة للوصول إليها من HTML
window.exportAsText = exportAsText;
window.exportAsJSON = exportAsJSON;
window.exportAsCSV = exportAsCSV;